#ifndef __CControlRoot__
#define __CControlRoot__

#include <Collections/TCountedPointerArray.hpp>
#include <Exceptions/CException.hpp>
#include "../Events/CMouseEvent.hpp"
#include "IControlRoot.hpp"
#include "IControl.hpp"

//	===========================================================================

using Exponent::Collections::TCountedPointerArray;
using Exponent::Exceptions::CException;
using Exponent::GUI::Events::CMouseEvent;
using Exponent::GUI::Controls::IControlRoot;
using Exponent::GUI::Controls::IControl;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @class CControlRoot CControlRoot.hpp
			 * @brief Implementaton to an control root (control container)
			 *
			 * @date 24/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @version 1.0.1 Added isControlAttatched function
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CControlRoot.hpp,v 1.6 2007/02/27 19:47:11 paul Exp $
			 */
			class CControlRoot : public IControlRoot
			{
			public:

				/**
				 * Construction
				 * @param parent The window that this control root is attatched to
				 */
				CControlRoot(IWindow *parent);

				/**
				 * Destruction
				 */
				virtual ~CControlRoot();

//	===========================================================================

				/**
				 * Lock a control (only this control recieves mouse info)
				 * @param control The control to feed all events to
				 */
				virtual void lockControl(IControl *control);

				/**
				 * Unlock a control
				 * @see lockControl
				 */
				virtual void unlockControl();

				/**
				 * Is there a control locked
				 * @retval bool True if a control is locked
				 */
				virtual bool controlIsLocked() const;

				/**
				 * Get the locked control
				 * @retval IControl* The locked control or NULL
				 */
				virtual IControl *getLockedControl() const;

//	===========================================================================

				/**
				 * Set the parent window
				 * @param parent The parent window
				 */
				virtual void setParentWindow(IWindow *parent);

				/**
				 * Get the parent window
				 * @retval IWindow* The parent window
				 */
				virtual IWindow *getParentWindow() const;
				
				/**
				 * Get window offset
				 * @retval CPoint The window offset for this panel
				 */
				virtual CPoint getWindowOffset();

				/**
				 * Is attatched to VST window
				 * @param attatched True if this window is attatched to a VST window, false otherwise
				 */
				virtual void isVSTAttatched(const bool attatched = true) { m_isVSTAttatched = attatched; }

//	===========================================================================

				/**
				 * Get the control at this point (point is relative to this' top left)
				 * @param point The point to check at
				 * @retval IControl* The control, or NULL on error or none
				 */
				virtual IControl *getControlAtPoint(const CPoint &point);		

				/**
				 * Get the control at this point, checks for mouse enabling
				 * @param point The point to check at
				 * @retval IControl* The control, or NULL on error or none
				 */
				virtual IControl *getMouseEnabledControlAtPoint(const CPoint &point);

				/**
				 * Get he control at this point, checks for drop file enabling
				 * @param point The point to check at
				 * @retval IControl* The control, or NULL on error or none
				 */
				virtual IControl *getDropEnabledControlAtPoint(const CPoint &point);

				/**
				 * Get the key enabled control at this point, checks for key handling
				 * @param point The point to check at
				 * @retval IControl* The control, or NULL on error or none
				 */
				virtual IControl *getKeyEnabledControlAtPoint(const CPoint &point);

//	===========================================================================

				/**
				 * Add a control
				 * @param control A control that becomes a child of this
				 */
				virtual void addControl(IControl *control);

				/**
				 * Remove a control
				 * @param control The control to remove from the child list
				 */
				virtual void removeControl(IControl *control);

				/**
				 * Remove all controls
				 * @see removeControl
				 */
				virtual void clearControls();

				/**
				 * Get a control at a specific index or uniqueId
				 * @param index The index of the control within the array or if isIndex is false then matches the unique id of the control
				 * @param isIndex If true then index is treated as an index, if is false then index is treated as control id
				 * @see IControl::getUniqueId()
				 * @retval IControl* The control or NULL on error
				 */
				virtual IControl *getControlAtIndex(const long index, bool isIndex = true);		

				/**
				 * Get the index of a control
				 * @retval long The index of the control passed, or TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER if not found
				 * @see TCountedPointerArray
				 * @see TPointerCollection
				 */
				virtual long getControlIndex(IControl *control);

				/**
				 * Get the number of controls embedded in this root
				 * @retval long The number of controls embedded in this root
				 */
				virtual long getNumberOfControls();

				/**
				 * Get the number of controls embedded in this root, actually inserted <= getNumberOfControls
				 * @retval long The array insert index
				 */
				virtual long getNumberOfInsertedControls();

				/**
				 * Is a certain control attatched?
				 * @param control The control to check if its attatched to this root
				 * @retval bool True if the control is attatched, false otherwise
				 */
				virtual bool isControlAttatched(IControl *control);

				/**
				 * Get the last used control
				 * @retval const IControl* The last control that sucessfully receieved mouse commands
				 */
				virtual const IControl *getLastUsedControl() const { return m_lastControl; }

//	===========================================================================

				/**
				 * Draw the root control
				 * @param graphics The graphics context
				 */
				virtual void drawRootControl(CGraphics &graphics);

				/**
				 * Update (redraw) a control
				 * @param control The control to redraw
				 */
				virtual void updateControl(IControl *control);

				/**
				 * Update (redraw) a control at a specific index, or with a specific id if isIndex = false
				 * @param index The index of the control within the array or if isIndex is false then matches the unique id of the control
				 * @param isIndex If true then index is treated as an index, if is false then index is treated as control id
				 * @see IControl::getUniqueId()
				 */
				virtual void updateControl(const long index, bool isIndex);

				/**
				 * Update an area
				 * @param area The area to update relative to our top left
				 */
				virtual void updateArea(const CRect &area);

//	===========================================================================

				/**
				 * Set the background image
				 * @param image The image to use for the background of the root control
				 */
				virtual void setBackgroundImage(IImage *image);

//	===========================================================================

				/**
				 * Start the timer to work out if the tool tip should fire up
				 */
				virtual void startToolTipTimer();

				/**
				 * Stop the tool tip timer from running
				 */
				virtual void stopToolTipTimer() ;

//	===========================================================================

				// Save or load a layout of this panel..
				//virtual void serialise(const EControlRootSerialiseOption option);

//	===========================================================================

				/**
				 * Handle left button being clicked
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonDown(CMouseEvent &event);

				/**
				 * Handle left button being released
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonUp(CMouseEvent &event);

				/**
				 * Handle the right button being clicked
				 * @param event The event to handle
				 */
				virtual void handleRightButtonDown(CMouseEvent &event);

				/**
				 * Handle the right button being released
				 * @param event The event to handle
				 */
				virtual void handleRightButtonUp(CMouseEvent &event);

				/**
				 * Handle a double click on the left button
				 * @param event The event to handle
				 */
				virtual void handleDoubleClick(CMouseEvent &event);

				/**
				 * Handle the scroll wheel
				 * @param event The event to handle
				 */
				virtual void handleMouseScroll(CMouseEvent &event);

				/**
				 * Handle the mouse movement
				 * @param event The event to handle
				 */
				virtual void handleMouseMovement(CMouseEvent &event);

				/**
				 * Handle a mouse leaving movement
				 * @param event The event to handle
				 */
				virtual void handleMouseLeavingArea(CMouseEvent &event);

//	===========================================================================

				/**
				 * Handle key down events
				 * @param event The event to handle
				 */
				virtual bool handleKeyDown(const CKeyboardEvent &event);
				
				/**
				 * Handle key up events
				 * @param event The event to handle
				 */
				virtual bool handleKeyUp(const CKeyboardEvent &event);

//	===========================================================================

				/**
				 * Handle a file drop
				 * @param event The event to handle
				 */
				virtual void handleFileDrop(const CDropEvent &event);

//	===========================================================================

				/**
				 * Get the global location of the control
				 * @param control The control to get the global coordinates of
				 * @param point On return contains the global position of the control
				 */
				virtual void getGlobalCoordinatesOfControl(IControl *control, CPoint &point);

				/**
				 * Get the window (absoloute) position of a control
				 * @param control The control to find the absolute position of
				 * @param point The position of the control on return
				 */
				virtual void getWindowCoordinatesOfControl(IControl *control, CPoint &point);

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Do all pre processing on a mouse event
				 * @param event The event to process
				 * @retval IControl* The control that should recieve the event or NULL if none
				 */
				IControl *preProcessMouseEvent(CMouseEvent &event);

				/**
				 * Do all post processing on a mouse event
				 * @param theControl The control to post mouse event process
				 */
				void postProcessMouseEvent(IControl *theControl);

//	===========================================================================

				TCountedPointerArray<IControl> *m_controlArray;		/**< The array of child controls */
				IControl *m_lockedControl;							/**< The current control */
				IControl *m_lastControl;							/**< The last control clicked */
				IImage *m_controlRootBackgroundImage;				/**< The background image */
				IWindow *m_parentWindow;							/**< The parent window */
				CMouseEvent m_subMouseEvent;						/**< The sub mouse event */
				CPoint m_originalPosition;							/**< The original position for mouse events */
				bool m_isVSTAttatched;								/**< Are we attatched to a VST window */
				
			};
		}
	}
}
#endif	// End of CControlRoot.hpp